unit PhryCheckBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, StdCtrls;

type
  TPhryCheckBox = class(TGraphicControl)
  private
    { Private declarations }
    FAlignment :TLeftRight;
    FState :TCheckBoxState;
    FAllowGrayed :Boolean;
    FTransparent :Boolean;
    FOnMouseClick :TNotifyEvent;
    FBitmaps :Array[TCheckBoxState] of TBitmap;
    procedure SetAlignment(Value :TLeftRight);
    procedure SetState(Value :TCheckBoxState);
    function GetChecked :Boolean;
    procedure SetChecked(Value :Boolean);
    procedure HandleMouseClick(Sender :TObject);
    procedure PaintBox(Surface :TCanvas; PaintRect :TRect);
    procedure PaintText(Surface :TCanvas; PaintRect :TRect);
    procedure SetBitmap(Index :TCheckBoxState; Value :TBitmap);
  protected
    { Protected declarations }
    procedure Paint; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    property Bitmaps[index :TCheckBoxState] :TBitmap write SetBitmap;
    procedure Blit(Surface :TCanvas; R :TRect);
  published
    { Published declarations }
    property Action;
    property Alignment: TLeftRight read FAlignment write SetAlignment default taRightJustify;
    property AllowGrayed: Boolean read FAllowGrayed write FAllowGrayed default False;
    property Anchors;
    property BiDiMode;
    property Caption;
    property Checked: Boolean read GetChecked write SetChecked default False;
    property Color;
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property Height default 17;
    property Width default 97;
    property ParentBiDiMode;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property State: TCheckBoxState read FState write SetState default cbUnchecked;
    property Transparent :Boolean read FTransparent write FTransparent default False;
    property Visible;
    property OnMouseClick :TNotifyEvent read FOnMouseClick write FOnMouseClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure Register;

implementation

const
  BoxWidth = 18;
  bufBitmaps:Array[TCheckBoxState,1..222] of byte = ((
66,77,222,0,0,0,0,0,0,0,118,0,0,0,40,0,
0,0,13,0,0,0,13,0,0,0,1,0,4,0,0,0,
0,0,104,0,0,0,196,14,0,0,196,14,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,128,0,0,128,
0,0,0,128,128,0,128,0,0,0,128,0,128,0,128,128,
0,0,192,192,192,0,128,128,128,0,0,0,255,0,0,255,
0,0,0,255,255,0,255,0,0,0,255,0,255,0,255,255,
0,0,255,255,255,0,255,255,255,255,255,255,240,0,135,119,
119,119,119,119,240,0,128,255,255,255,255,247,240,0,128,255,
255,255,255,247,240,0,128,255,255,255,255,247,240,0,128,255,
255,255,255,247,240,0,128,255,255,255,255,247,240,0,128,255,
255,255,255,247,240,0,128,255,255,255,255,247,240,0,128,255,
255,255,255,247,240,0,128,255,255,255,255,247,240,0,128,0,
0,0,0,7,240,0,136,136,136,136,136,136,240,0),(
66,77,222,0,0,0,0,0,0,0,118,0,0,0,40,0,
0,0,13,0,0,0,13,0,0,0,1,0,4,0,0,0,
0,0,104,0,0,0,196,14,0,0,196,14,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,128,0,0,128,
0,0,0,128,128,0,128,0,0,0,128,0,128,0,128,128,
0,0,192,192,192,0,128,128,128,0,0,0,255,0,0,255,
0,0,0,255,255,0,255,0,0,0,255,0,255,0,255,255,
0,0,255,255,255,0,255,255,255,255,255,255,240,0,135,119,
119,119,119,119,240,0,128,255,255,255,255,247,240,0,128,255,
240,255,255,247,240,0,128,255,0,15,255,247,240,0,128,240,
0,0,255,247,240,0,128,240,15,0,15,247,240,0,128,240,
255,240,0,247,240,0,128,255,255,255,0,247,240,0,128,255,
255,255,240,247,240,0,128,255,255,255,255,247,240,0,128,0,
0,0,0,7,240,0,136,136,136,136,136,136,240,0),(
66,77,222,0,0,0,0,0,0,0,118,0,0,0,40,0,
0,0,13,0,0,0,13,0,0,0,1,0,4,0,0,0,
0,0,104,0,0,0,196,14,0,0,196,14,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,128,0,0,128,
0,0,0,128,128,0,128,0,0,0,128,0,128,0,128,128,
0,0,192,192,192,0,128,128,128,0,0,0,255,0,0,255,
0,0,0,255,255,0,255,0,0,0,255,0,255,0,255,255,
0,0,255,255,255,0,255,255,255,255,255,255,240,0,135,119,
119,119,119,119,240,0,128,247,247,247,247,247,240,0,128,127,
120,127,127,119,240,0,128,247,136,135,247,247,240,0,128,120,
136,136,127,119,240,0,128,248,135,136,135,247,240,0,128,120,
127,120,136,119,240,0,128,247,247,247,136,247,240,0,128,127,
127,127,120,119,240,0,128,247,247,247,247,247,240,0,128,0,
0,0,0,7,240,0,136,136,136,136,136,136,240,0));

constructor TPhryCheckBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  FState := cbUnChecked;
  FAllowGrayed := False;
  FTransparent := False;
  OnClick := HandleMouseClick;
  OnDblClick := HandleMouseClick;
  Height := 17;
  Width := 97;
  FBitmaps[cbUnChecked] := TBitmap.Create;
  FBitmaps[cbChecked] := TBitmap.Create;
  FBitmaps[cbGrayed] := TBitmap.Create;
  Bitmaps[cbUnChecked] := nil;
  Bitmaps[cbChecked] := nil;
  Bitmaps[cbGrayed] := nil;
end;

procedure TPhryCheckBox.SetBitmap(Index :TCheckBoxState; Value :TBitmap);
var
  s :TMemoryStream;
begin
  if Value = nil then
  begin
    s := TMemoryStream.Create;
    s.SetSize(sizeof(bufBitmaps[Index]));
    s.Seek(0, soFromBeginning);
    s.Write(bufBitmaps[Index], Sizeof(bufBitmaps[Index]));
    s.Seek(0, soFromBeginning);
    FBitmaps[Index].LoadFromStream(s);
  end else
    FBitmaps[Index].Canvas.CopyRect(FBitmaps[Index].Canvas.ClipRect, Value.Canvas, FBitmaps[Index].Canvas.ClipRect);
end;

function TPhryCheckBox.GetChecked: Boolean;
begin
  Result := State = cbChecked;
end;

procedure TPhryCheckBox.SetAlignment(Value: TLeftRight);
begin
  if FAlignment <> Value then
  begin
    FAlignment := Value;
    Refresh;
  end;
end;

procedure TPhryCheckBox.SetChecked(Value: Boolean);
begin
  if Value then State := cbChecked else State := cbUnchecked;
end;

procedure TPhryCheckBox.SetState(Value: TCheckBoxState);
begin
  if FState <> Value then
  begin
    FState := Value;
    Refresh;
  end;
end;

procedure TPhryCheckBox.HandleMouseClick(Sender :TObject);
begin
  case FState of
    cbUnChecked:
      if FAllowGrayed then
        FState := cbGrayed
      else
        FState := cbChecked;

    cbGrayed:
      FState := cbChecked;

    cbChecked:
      FState := cbUnchecked;
  end;
  Refresh;
  if Assigned(FOnMouseClick) then
    FOnMouseClick(Sender);
end;

procedure TPhryCheckBox.Paint;
begin
  Blit(Canvas, Rect(0, 0, Width - 1, Height - 1));
end;

procedure TPhryCheckBox.Blit(Surface :TCanvas; R :TRect);
var
  TheImage :TBitmap;
  PaintRect: TRect;
begin
  TheImage := TBitmap.Create;
  try
    TheImage.Height := R.Bottom - R.Top + 1;
    TheImage.Width := R.Right - R.Left + 1;
    PaintRect := TheImage.Canvas.ClipRect;
    TheImage.Canvas.CopyMode := cmSrcCopy;
    TheImage.Canvas.Brush.Color := Color;
    if FTransparent then
      TheImage.Canvas.CopyRect(PaintRect, Canvas, PaintRect)
    else
      TheImage.Canvas.FillRect(PaintRect);
    PaintBox(TheImage.Canvas, PaintRect);
    if FAlignment = taLeftJustify then
      PaintRect.Left := PaintRect.Left + BoxWidth
    else
      PaintRect.Right := PaintRect.Right - BoxWidth;
    PaintText(TheImage.Canvas, PaintRect);
    Surface.CopyMode := cmSrcCopy;
    Surface.Draw(0, 0, TheImage);
  finally
    TheImage.Free;
  end;
end;

procedure TPhryCheckBox.PaintBox(Surface :TCanvas; PaintRect :TRect);
var
  X, Y :Longint;
begin
  X := PaintRect.Left + 1;
  Y := PaintRect.Top + (PaintRect.Bottom - PaintRect.Top - 13) div 2;
  Surface.CopyMode := cmSrcCopy;
  Surface.Draw(X, Y, FBitmaps[FState]);
end;

procedure TPhryCheckBox.PaintText(Surface :TCanvas; PaintRect :TRect);
var
  X, Y :Longint;
begin
  Surface.Brush.Style := bsClear;
  Surface.Font := Font;
  Y := PaintRect.Top + (PaintRect.Bottom - PaintRect.Top - Surface.TextHeight(Caption)) div 2;
  if FAlignment = taLeftJustify then
    X := PaintRect.Left
  else
    X := PaintRect.Right - Surface.TextWidth(Caption);
  Surface.TextRect(PaintRect, X, Y, Caption);
end;

procedure Register;
begin
  RegisterComponents('Phry', [TPhryCheckBox]);
end;

end.
